<?php defined('BASEPATH') OR exit('No direct script access allowed');

class Product extends CI_Controller
{
    public function __construct(){
        parent::__construct();
        if (!$this->session->userdata('admin_id')) redirect('admin/login');
        $this->load->model('Product_model','product');
        $this->load->helper(['url','file','form','security']);
        $this->load->library(['form_validation','upload','image_lib','session']);
    }

    private function render($view,$data=[]){
        $this->load->view('admin/layouts/header',$data);
        $this->load->view($view,$data);
        $this->load->view('admin/layouts/footer');
    }

    public function index(){
        $data['title'] = 'Kelola Produk';
        $data['items'] = $this->product->all();
        $this->render('admin/product_index', $data);
    }

    public function create(){
        $data = [
            'title'        => 'Tambah Produk',
            'row'          => null,
            'upload_error' => ''
        ];

        if ($this->input->method()==='post'){
            $this->form_validation->set_rules('name','Nama Produk','required|min_length[3]');
            $this->form_validation->set_rules('kode','Nomor/Kode Produk','required|alpha_dash|is_unique[products.kode]');
            $this->form_validation->set_rules('price','Harga','required');

            if ($this->form_validation->run()){
                if (empty($_FILES['image']['name'])){
                    $data['upload_error'] = 'Gambar wajib diupload.';
                    return $this->render('admin/product_form',$data);
                }

                $this->upload->initialize([
                    'upload_path'   => './assets/img/products/',
                    'allowed_types' => 'jpg|jpeg|png|webp',
                    'max_size'      => 5096, // 4MB
                    'encrypt_name'  => TRUE,
                ]);

                if ($this->upload->do_upload('image')){
                    $filename = $this->upload->data('file_name');
                    $this->_resize_main($filename);
                    $this->_make_thumb($filename);

                    $insert = [
                        'name'        => html_escape($this->input->post('name',TRUE)),
                        'kode'        => html_escape($this->input->post('kode',TRUE)),
                        'description' => html_escape($this->input->post('description',TRUE)),
                        'keterangan'  => html_escape($this->input->post('keterangan',TRUE)),
                        'price'       => $this->_to_decimal($this->input->post('price', TRUE)),
                        'image'       => $filename,
                        // TIDAK menulis created_at
                    ];
                    $this->product->insert($insert);
                    $this->session->set_flashdata('ok','Produk tersimpan.');
                    return redirect('admin/produk');
                } else {
                    $data['upload_error'] = $this->upload->display_errors('','');
                    return $this->render('admin/product_form',$data);
                }
            }
        }

        $this->render('admin/product_form',$data);
    }

    public function edit($id){
        $row = $this->product->get($id);
        if(!$row) show_404();

        $data = [
            'title'        => 'Edit Produk',
            'row'          => $row,
            'upload_error' => ''
        ];

        if ($this->input->method()==='post'){
            $this->form_validation->set_rules('name','Nama Produk','required|min_length[3]');
            $this->form_validation->set_rules('kode','Nomor/Kode Produk','required|alpha_dash');
            $this->form_validation->set_rules('price','Harga','required');

            if ($this->form_validation->run()){
                $postedKode = $this->input->post('kode', TRUE);
                if ($this->_kode_exists($postedKode, $id)){
                    $this->session->set_flashdata('err','Kode produk sudah digunakan. Gunakan kode lain.');
                    return $this->render('admin/product_form',$data);
                }

                $update = [
                    'name'        => html_escape($this->input->post('name',TRUE)),
                    'kode'        => html_escape($postedKode),
                    'description' => html_escape($this->input->post('description',TRUE)),
                    'keterangan'  => html_escape($this->input->post('keterangan',TRUE)),
                    'price'       => $this->_to_decimal($this->input->post('price', TRUE)),
                    // TIDAK menulis updated_at
                ];

                if (!empty($_FILES['image']['name'])){
                    $this->upload->initialize([
                        'upload_path'   => './assets/img/products/',
                        'allowed_types' => 'jpg|jpeg|png|webp',
                        'max_size'      => 4096,
                        'encrypt_name'  => TRUE,
                    ]);

                    if ($this->upload->do_upload('image')){
                        $filename = $this->upload->data('file_name');
                        $this->_resize_main($filename);
                        $this->_make_thumb($filename);
                        $update['image'] = $filename;

                        if(!empty($row->image)) $this->_delete_image_files($row->image);
                    } else {
                        $data['upload_error'] = $this->upload->display_errors('','');
                        return $this->render('admin/product_form',$data);
                    }
                }

                $this->product->update($id,$update);
                $this->session->set_flashdata('ok','Perubahan disimpan.');
                return redirect('admin/produk');
            }
        }

        $this->render('admin/product_form',$data);
    }

    public function delete($id){
        $row = $this->product->get($id);
        if($row){
            if(!empty($row->image)) $this->_delete_image_files($row->image);
            $this->product->delete($id);
            $this->session->set_flashdata('ok','Produk dihapus.');
        }
        redirect('admin/produk');
    }

    // alias biar URL /hapus/{id} tetap berfungsi
    public function hapus($id){ return $this->delete($id); }

    // ===== Helpers =====
    private function _kode_exists($kode, $exclude_id=null){
        $this->db->from('products')->where('kode',$kode);
        if ($exclude_id) $this->db->where('id !=', $exclude_id);
        return $this->db->count_all_results() > 0;
    }

    private function _to_decimal($s){
        $s = str_replace([' ','Rp','rp','.'], '', $s);
        $s = str_replace(',', '.', $s);
        return is_numeric($s) ? (float)$s : 0;
    }

    private function _resize_main($filename){
        $src = FCPATH.'assets/img/products/'.$filename;
        $cfg = [
            'image_library'  => 'gd2',
            'source_image'   => $src,
            'new_image'      => $src,
            'maintain_ratio' => TRUE,
            'width'          => 1200,
            'height'         => 1200,
            'quality'        => '85%'
        ];
        $this->image_lib->initialize($cfg);
        $this->image_lib->resize();
        $this->image_lib->clear();
    }

    private function _make_thumb($filename){
        $src   = FCPATH.'assets/img/products/'.$filename;
        $pi    = pathinfo($src);
        $thumb = $pi['dirname'].'/'.$pi['filename'].'_thumb.'.$pi['extension'];
        @copy($src, $thumb);

        $size = @getimagesize($thumb);
        if ($size){
            $w=$size[0]; $h=$size[1]; $target_ratio=360/240; $img_ratio=$h==0?1:$w/$h;
            $cfg=['image_library'=>'gd2','source_image'=>$thumb,'new_image'=>$thumb,'maintain_ratio'=>TRUE];
            if($img_ratio > $target_ratio) $cfg['height']=240; else $cfg['width']=360;
            $this->image_lib->initialize($cfg); $this->image_lib->resize(); $this->image_lib->clear();
        }
        $size=@getimagesize($thumb);
        if($size){
            $w=$size[0]; $h=$size[1]; $x=max(0,intval(($w-360)/2)); $y=max(0,intval(($h-240)/2));
            $cfg=['image_library'=>'gd2','source_image'=>$thumb,'new_image'=>$thumb,'maintain_ratio'=>FALSE,'width'=>360,'height'=>240,'x_axis'=>$x,'y_axis'=>$y];
            $this->image_lib->initialize($cfg); $this->image_lib->crop(); $this->image_lib->clear();
        }
    }

    private function _delete_image_files($filename){
        $main  = FCPATH.'assets/img/products/'.$filename;
        $pi    = pathinfo($main);
        $thumb = $pi['dirname'].'/'.$pi['filename'].'_thumb.'.$pi['extension'];
        @unlink($main);
        @unlink($thumb);
    }
}
